<?php

namespace WP_Defender\Model\Setting;

use Calotes\Model\Setting;

class Events extends Setting {
	/**
	 * Option name
	 * @var string
	 */
	public $table = 'wd_audit_cached';
	/**
	 * The active data
	 * @var array
	 */
	public $data = [];
	/**
	 * Last data, use for backup
	 * @var array
	 */
	public $old_data = [];
	/**
	 * This is pending to on cloud
	 * @var array
	 */
	public $events_pending = [];
	/**
	 * Timestamp last sync
	 * @var int
	 */
	public $last_sync;
	/**
	 * Date from
	 * @var
	 */
	public $last_sync_from;

	/**
	 * @param array $params
	 *
	 * @return int
	 */
	public function has_data( $params = [] ) {
		if ( ! is_countable( $this->data ) ) {
			//fail safe
			return false;
		}
		/**
		 * if params empty means we query for summary, which mostly cached
		 */
		if ( empty( $params ) && is_countable( $this->data ) ) {
			return count( $this->data );
		}
		/**
		 * because we only cached for 2 months, so if user pick older there wont be data
		 * return from API instead
		 */
		if ( is_countable( $this->data ) ) {

			//if the timestamp smaller date from value, means we will have to pull the stuff from API
			if ( strtotime( $params['date_from'] ) > strtotime( $this->last_sync_from ) ) {
				return true;
			}
		}

		//fail safe
		return false;
	}

	/**
	 *  This will fetch data from cloud and store on local
	 */
	public function fetch() {
		//we will fetch the data of recent 3 months, the cached will be pulled daily
		//if user fetch things that we don't have in local cache, fetch and merge
		$date_format = 'Y-m-d H:i:s';
		$args        = [
			'date_from' => date( $date_format, strtotime( 'midnight', strtotime( '-30 days', current_time( 'timestamp' ) ) ) ),
			'date_to'   => date( $date_format, strtotime( 'tomorrow', current_time( 'timestamp' ) ) ),
		];
		$this->log( sprintf( 'Fetching data from %s to %s to local', $args['date_from'], $args['date_to'] ) );
		$data = \WP_Defender\Component\Audit\Audit_API::pull_logs( $args, 'timestamp', 'desc', true );

		if ( is_wp_error( $data ) ) {
			$this->log( sprintf( 'Fetch error: %s', $data->get_error_message() ) );

			return;
		}
		if ( is_array( $data ) && $data['status'] == 'success' ) {
			//backup all the old time
			$this->old_data     = array_merge( $this->data, $this->events_pending );
			$this->data         = $data['data'];
			$this->last_sync     = time();
			$this->last_sync_from = $args['date_from'];
			$this->log( sprintf( 'Fetched done. %s records', count( $data['data'] ) ) );
		}
		$this->save();
	}
	/**
	 *
	 *
	 * @param array $filter
	 * @param string $order_by
	 * @param string $order
	 * @param bool $nopaging
	 *
	 * @return array
	 */
	public function get_data( $filter = array(), $order_by = 'timestamp', $order = 'desc', $nopaging = false ) {
		$data = $this->filter_data( array_merge( $this->data, $this->events_pending ), $filter );
		usort( $data, function ( $a, $b ) use ( $order, $order_by ) {

			if ( $order == 'desc' ) {
				return (int) $b[ $order_by ] > (int) $a[ $order_by ];
			} else {
				return (int) $a[ $order_by ] > (int) $b[ $order_by ];
			}
		} );

		$per_page = 40;

		return array(
			'data'        => $data,
			'total_items' => count( $data ),
			'total_pages' => ceil( count( $data ) / $per_page ),
			'per_page'    => $per_page
		);
	}

	/**
	 * @param $data
	 * @param array $filter
	 *
	 * @return array
	 */
	private function filter_data( $data, $filter = [] ) {
		/**
		 * data can be filter from
		 * date range
		 * user_id
		 * event type
		 * ip
		 * context
		 * action type
		 */
		$date_from = strtotime( $filter['date_from'] );
		$date_to   = strtotime( $filter['date_to'] );
		foreach ( $data as $i => $item ) {
			if ( $item['timestamp'] < $date_from || $item['timestamp'] > $date_to ) {
				unset( $data[ $i ] );
				continue;
			}

			if ( ! empty( $filter['event_type'] ) && ! in_array( $item['event_type'], $filter['event_type'] ) ) {
				unset( $data[ $i ] );
				continue;
			}

			if ( ! empty( $filter['user_id'] ) && $item['user_id'] != $filter['user_id'] ) {
				unset( $data[ $i ] );
				continue;
			}

			if ( ! empty( $filter['ip'] ) && ! stristr( $item['ip'], $filter['ip'] ) ) {
				unset( $data[ $i ] );
				continue;
			}

		}

		return $data;
	}
}