<?php

namespace WP_Defender\Controller;

use Calotes\Component\Request;
use Calotes\Component\Response;
use Calotes\Helper\HTTP;
use Calotes\Helper\Route;
use WP_Defender\Behavior\WPMUDEV;
use WP_Defender\Controller2;
use WP_Defender\Model\Lockout_Log;
use WP_Defender\Model\Notification\Audit_Report;
use WP_Defender\Model\Notification\Firewall_Notification;
use WP_Defender\Model\Notification\Firewall_Report;
use WP_Defender\Model\Notification\Malware_Notification;
use WP_Defender\Model\Notification\Malware_Report;
use WP_Defender\Model\Notification\Tweak_Reminder;
use WP_Defender\Model\Setting\Blacklist_Lockout;
use WP_Defender\Model\Setting\Login_Lockout;
use WP_Defender\Model\Setting\Notfound_Lockout;
use WP_Defender\Model\Setting\Two_Fa;
use WP_Defender\Traits\Formats;
use WP_Defender\Traits\IO;

/**
 * This class will use to create a main admin page
 *
 * Class Dashboard
 * @package WP_Defender\Dashboard\Controller
 * @method bool is_pro
 */
class Dashboard extends Controller2 {
	use IO, Formats;

	const CACHE_BLACKLIST_STATUS = 'wpdefender_blacklist_status', CACHE_TIME = 300;

	public $slug = 'wp-defender';

	public function __construct() {
		$this->attach_behavior( WPMUDEV::class, WPMUDEV::class );
		$this->add_main_page();
		$this->register_routes();
		add_action( 'defender_enqueue_assets', [ &$this, 'enqueue_assets' ] );
		add_action( 'wdp_register_hub_action', [ &$this, 'add_hub_endpoint' ] );
		add_filter( 'custom_menu_order', '__return_true' );
		add_filter( 'menu_order', [ &$this, 'menu_order' ] );
		add_action( 'defender_hub_sync', [ &$this, 'hub_sync' ] );
		add_action( 'admin_init', [ &$this, 'maybe_redirect_notification_request' ], 99 );
	}

	/**
	 * Because we move the notifications on seprate modules, so links from HUB should be redirect to correct URL
     * @return void
	 */
	public function maybe_redirect_notification_request() {
		$page = HTTP::get( 'page' );
		if ( ! in_array( $page, [ 'wdf-scan', 'wdf-ip-lockout', 'wdf-hardener', 'wdf-logging' ] ) ) {
			return;
		}
		$view = HTTP::get( 'view' );
		if ( in_array( $view, [ 'reporting', 'notification', 'report' ] ) ) {
			wp_redirect( network_admin_url( 'admin.php?page=wdf-notification' ) );
			exit;
		}

		return;
	}

	/**
	 * Filter out the defender menu for changing text
	 *
	 * @param $menu_order
	 *
	 * @return mixed
	 */
	public function menu_order( $menu_order ) {
		global $submenu;
		if ( isset( $submenu['wp-defender'] ) ) {
			$defender_menu          = $submenu['wp-defender'];
			$defender_menu[0][0]    = esc_html__( 'Dashboard', 'wpdef' );
			$defender_menu          = array_values( $defender_menu );
			$submenu['wp-defender'] = $defender_menu;
		}

		global $menu;
		//$count     = $this->countTotalIssues();
		/**
		 * Indicator is the total of tweaks & scanning issues
		 */
		$count = 0;
		$scan  = \WP_Defender\Model\Scan::get_last();
		if ( is_object( $scan ) ) {
			$count = count( $scan->get_issues() );
		}
		$tweaks    = new \WP_Defender\Model\Setting\Security_Tweaks();
		$count     = $count + count( $tweaks->issues );
		$indicator = $count > 0 ? ' <span class="update-plugins wd-issue-indicator-sidebar"><span class="plugin-count">' . $count . '</span></span>' : null;
		foreach ( $menu as $k => $item ) {
			if ( 'wp-defender' === $item[2] ) {
				$menu[ $k ][0] .= $indicator;
			}
		}

		return $menu_order;
	}

	public function add_hub_endpoint( $actions ) {
		$actions['defender_new_scan']          = array( &$this, 'new_scan' );
		$actions['defender_schedule_scan']     = array( &$this, 'schedule_scan' );
		$actions['defender_manage_audit_log']  = array( &$this, 'manage_audit_log' );
		$actions['defender_manage_lockout']    = array( &$this, 'manage_lockout' );
		$actions['defender_whitelist_ip']      = array( &$this, 'whitelist_ip' );
		$actions['defender_blacklist_ip']      = array( &$this, 'blacklist_ip' );
		$actions['defender_get_stats']         = array( &$this, 'get_stats' );
		$actions['defender_get_scan_progress'] = array( &$this, 'get_scan_progress' );

		//backup/restore settings
		$actions['defender_export_settings'] = array( &$this, 'export_settings' );
		$actions['defender_import_settings'] = array( &$this, 'import_settings' );
		//get stats
		$actions['defender_get_stats_v2'] = [ &$this, 'defender_get_stats_v2' ];

		return $actions;
	}

	public function defender_get_stats_v2() {
		global $wp_version;
		$audit = wd_di()->get( Audit_Logging::class )->summary_data();
		$scan  = \WP_Defender\Model\Scan::get_last();
		$total = 0;
		if ( is_object( $scan ) ) {
			$total += count( $scan->get_issues() );
		}
		$tweaks = wd_di()->get( Security_Tweaks::class )->data_frontend();
		$total  += $tweaks['summary']['issues_count'];
		$ret    = [
			'summary'         => [
				'count'     => $total,
				'next_scan' => wd_di()->get( Malware_Report::class )->get_next_run_as_string()
			],
			'report'          => [
				'malware_scan'  => wd_di()->get( Malware_Report::class )->get_next_run_as_string( true ),
				'firewall'      => wd_di()->get( Firewall_Report::class )->get_next_run_as_string( true ),
				'audit_logging' => wd_di()->get( Audit_Report::class )->get_next_run_as_string( true )
			],
			'security_tweaks' => [
				'issues'       => $tweaks['summary']['issues_count'],
				'fixed'        => $tweaks['summary']['fixed_count'],
				'notification' => wd_di()->get( Tweak_Reminder::class )->status === \WP_Defender\Model\Notification::STATUS_ACTIVE,
				'wp_version'   => $wp_version,
				'php_version'  => phpversion()
			],
			'malware_scan'    => [
				'count'        => wd_di()->get( \WP_Defender\Model\Scan::class )->to_array()['count']['total'],
				'notification' => wd_di()->get( Malware_Notification::class )->status === \WP_Defender\Model\Notification::STATUS_ACTIVE
			],
			'firewall'        => [
				'last_lockout'        => Lockout_Log::get_last_lockout_date(),
				'24_hours'            => [
					'login_lockout' => Lockout_Log::count( strtotime( '-24 hours' ), time(), [
						Lockout_Log::AUTH_LOCK
					] ),
					'404_lockout'   => Lockout_Log::count( strtotime( '-24 hours' ), time(), [
						Lockout_Log::LOCKOUT_404
					] )
				],
				'7_days'              => [
					'login_lockout' => Lockout_Log::count_login_lockout_last_7_days(),
					'404_lockout'   => Lockout_Log::count_404_lockout_last_7_days()
				],
				'30_days'             => [
					'login_lockout' => Lockout_Log::count( strtotime( '-30 days' ), time(), [
						Lockout_Log::AUTH_LOCK
					] ),
					'404_lockout'   => Lockout_Log::count( strtotime( '-30 days' ), time(), [
						Lockout_Log::LOCKOUT_404
					] )
				],
				'notification_status' => [
					'login_lockout' => wd_di()->get( Firewall_Notification::class )->configs['login_lockout'],
					'404_lockout'   => wd_di()->get( Firewall_Notification::class )->configs['nf_lockout']
				]
			],
			'audit'           => [
				'last_event' => $audit['lastEvent'],
				'24_hours'   => $audit['dayCount'],
				'7_days'     => $audit['weekCount'],
				'30_days'    => $audit['monthCount']
			],
			'advanced_tools'  => [
				'security_headers' => [
					'sh_xframe'               => wd_di()->get( \WP_Defender\Model\Setting\Security_Headers::class )->sh_xframe,
					'sh_xss_protection'       => wd_di()->get( \WP_Defender\Model\Setting\Security_Headers::class )->sh_xss_protection,
					'sh_content_type_options' => wd_di()->get( \WP_Defender\Model\Setting\Security_Headers::class )->sh_content_type_options,
					'sh_strict_transport'     => wd_di()->get( \WP_Defender\Model\Setting\Security_Headers::class )->sh_strict_transport,
					'sh_referrer_policy'      => wd_di()->get( \WP_Defender\Model\Setting\Security_Headers::class )->sh_referrer_policy,
					'sh_feature_policy'       => wd_di()->get( \WP_Defender\Model\Setting\Security_Headers::class )->sh_feature_policy
				],
				'mask_login'       => wd_di()->get( \WP_Defender\Model\Setting\Mask_Login::class )->is_active()
			],
			'two_fa'          => [
				'status'     => wd_di()->get( Two_Fa::class )->enabled,
				'lost_phone' => wd_di()->get( Two_Fa::class )->lost_phone
			]
		];

		wp_send_json_success( [
			'stats' => $ret
		] );
	}

	/**
	 * Push scan data into HUB
	 */
	public function get_scan_progress() {
		$model = \WP_Defender\Model\Scan::get_active();
		if ( ! is_object( $model ) ) {
			wp_send_json_success(
				array(
					'progress' => - 1
				)
			);
		}
		$percent = $model->percent;
		if ( $percent > 100 ) {
			$percent = 100;
		}
		wp_send_json_success(
			array(
				'progress' => $percent,
			)
		);
	}

	/**
	 * Push data into HUB
	 *
	 * @param $params
	 * @param $action
	 */
	public function get_stats( $params, $action ) {
		$data = $this->build_stats_to_hub();
		wp_send_json_success(
			array(
				'stats' => $data,
			)
		);
	}

	public function blacklist_ip( $params, $action ) {
		$settings = new Blacklist_Lockout();
		$ip       = $params['ip'];
		if ( $ip && filter_var( $ip, FILTER_VALIDATE_IP ) ) {
			$settings->remove_from_list( $ip, 'allowlist' );
			$settings->add_to_list( $ip, 'blocklist' );
		} else {
			wp_send_json_error();
		}
		wp_send_json_success();
	}

	public function whitelist_ip( $params, $action ) {
		$settings = new Blacklist_Lockout();
		$ip       = $params['ip'];
		if ( $ip && filter_var( $ip, FILTER_VALIDATE_IP ) ) {
			$settings->remove_from_list( $ip, 'blocklist' );
			$settings->add_to_list( $ip, 'allowlist' );
		} else {
			wp_send_json_error();
		}
		wp_send_json_success();
	}

	public function manage_lockout( $params, $action ) {
		$type     = $params['type'];
		$response = array();
		if ( 'login' === $type ) {
			$settings = new Login_Lockout();
			if ( $settings->enabled ) {
				$settings->enabled = false;
				$response[ $type ] = 'disabled';
			} else {
				$settings->enabled = true;
				$response[ $type ] = 'enabled';
			}
			$settings->save();
		} elseif ( '404' === $type ) {
			$settings = new Notfound_Lockout();
			if ( $settings->enabled ) {
				$settings->enabled = false;
				$response[ $type ] = 'disabled';
			} else {
				$settings->enabled = true;
				$response[ $type ] = 'enabled';
			}
			$settings->save();
		} else {
			$response[ $type ] = 'invalid';
		}
		wp_send_json_success();
	}

	public function manage_audit_log() {
		$response = null;
		if ( class_exists( \WP_Defender\Model\Setting\Audit_Logging::class ) ) {
			$settings = new \WP_Defender\Model\Setting\Audit_Logging();
			$response = [];
			if ( true === $settings->enabled ) {
				$settings->enabled   = false;
				$response['enabled'] = false;
			} else {
				$settings->enabled   = true;
				$response['enabled'] = true;
			}
			$settings->save();
		}
		wp_send_json_success( $response );
	}

	/**
	 * Schedule a scan, from HUB
	 *
	 * @param $params
	 */
	public function schedule_scan( $params ) {
		$frequency    = $params['frequency'];
		$day          = $params['day'];
		$time         = $params['time'];
		$allowed_freq = array( 1, 7, 30 );
		if (
			! in_array( $frequency, $allowed_freq )
			|| ! in_array( $day, $this->get_days_of_week() )
			|| ! in_array( $time, $this->get_times() )
		) {
			wp_send_json_error();
		}
		$malware_report            = new Malware_Report();
		$malware_report->frequency = $frequency;
		$malware_report->day       = $day;
		$malware_report->time      = $time;
		$malware_report->save();

		wp_send_json_success();
	}

	/**
	 * Create new scan, triggered from HUB
	 */
	public function new_scan() {
		$scan = \WP_Defender\Model\Scan::create();
		if ( is_wp_error( $scan ) ) {
			wp_send_json_error( [
				'message' => $scan->get_error_message()
			] );
		}
		//Todo: need to save Malware_Report last_sent & est_timestamp?
		wd_di()->get( Scan::class )->self_ping();
		wp_send_json_success();
	}

	public function hub_sync() {
		$data = $this->build_stats_to_hub();
		$ret  = $this->make_wpmu_request( WPMUDEV::API_HUB_SYNC, $data, [
			'method' => 'POST'
		] );
	}

	/**
	 * Determine if we should show the quick setup
	 * This will show in those scenario
	 * 1. New setup
	 * 2. Just upgrade from free
	 *
	 * @return bool
	 * @defender_property
	 */
	public function maybe_show_quick_setup() {
		if ( get_site_transient( 'wp_defender_is_free_activated' ) == 1 ) {
			return 1;
		}

		//site just created
		if ( get_site_option( 'wp_defender_shown_activator' ) === false ) {
			return 1;
		}

		return 0;
	}

	protected function add_main_page() {
		$this->register_page( $this->get_menu_title(), $this->parent_slug, [
			&$this,
			'main_view'
		], null, $this->get_menu_icon() );
	}

	public function main_view() {
		$this->render( 'main' );
	}

	/**
	 * Endpoint for getting domain status
	 * @defender_route
	 */
	public function blacklist_status() {
		$status = get_site_transient( self::CACHE_BLACKLIST_STATUS );
		if ( false === $status ) {
			$status = $this->domain_status();
			set_site_transient( self::CACHE_BLACKLIST_STATUS, $status, self::CACHE_TIME );
		}

		if ( is_wp_error( $status ) ) {
			return new Response( false, [
				'status_error' => $status->get_error_message()
			] );
		}

		return new Response( true, [
			'status' => $status
		] );
	}

	/**
	 * Getting domain status
	 * @return int
	 */
	protected function domain_status() {
		$this->attach_behavior( WPMUDEV::class, WPMUDEV::class );
		$response = $this->make_wpmu_request( WPMUDEV::API_BLACKLIST );

		if ( is_wp_error( $response ) ) {
			if ( 412 === $response->get_error_code() ) {
				return - 1;
			}

			return $response;
		}
		$status = 1;
		foreach ( $response['services'] as $service ) {
			if ( true === $service['blacklisted'] ) {
				$status = 0;
			}
		}

		return $status;
	}

	/**
	 * @return Response
	 * @defender_route
	 */
	public function toggle_blacklist_status( Request $request ) {
		$data           = $request->get_data( [
			'status' => [
				'type'     => 'string',
				'sanitize' => 'sanitize_text_field'
			]
		] );
		$current_status = isset( $data['status'] ) ? $data['status'] : null;
		if ( ! in_array( $current_status, [ 'good', 'new', 'blacklisted' ] ) ) {
			return false;
		}

		$this->attach_behavior( WPMUDEV::class, WPMUDEV::class );
		if ( ! $this->is_pro() ) {
			return new Response( false, [
				'message' => __( "A WPMU DEV subscription is required for blocklist monitoring", 'wpdef' )
			] );
		}
		if ( 'new' === $current_status ) {
			$this->make_wpmu_request( WPMUDEV::API_BLACKLIST, [], [
				'method' => 'POST'
			] );
			$status = $this->domain_status();
		} else {
			$this->make_wpmu_request( WPMUDEV::API_BLACKLIST, [], [
				'method' => 'DELETE'
			] );
			$status = - 1;
		}
		set_site_transient( self::CACHE_BLACKLIST_STATUS, $status, self::CACHE_TIME );

		return new Response( true, [
			'status' => $status
		] );
	}

	/**
	 *
	 */
	public function enqueue_assets() {
		if ( ! $this->is_page_active() ) {
			return;
		}
		wp_localize_script( 'def-dashboard', 'dashboard', array_merge( $this->data_frontend(), $this->dump_routes_and_nonces() ) );
		wp_enqueue_script( 'def-dashboard' );
		$this->enqueue_main_assets();
	}

	/**
	 * @defender_route
	 */
	public function hide_new_features( Request $request ) {
		delete_site_option( 'wd_show_new_feature' );

		return new Response( true, [] );
	}

	/**
	 * Return svg image
	 * @return string
	 */
	private function get_menu_icon() {
		ob_start();
		?>
        <svg width="17px" height="18px" viewBox="10 397 17 18" version="1.1" xmlns="http://www.w3.org/2000/svg"
        >
            <!-- Generator: Sketch 3.8.3 (29802) - http://www.bohemiancoding.com/sketch -->
            <desc>Created with Sketch.</desc>
            <defs></defs>
            <path
                    d="M24.8009393,403.7962 L23.7971393,410.1724 C23.7395393,410.5372 23.5313393,410.8528 23.2229393,411.0532 L18.4001393,413.6428 L13.5767393,411.0532 C13.2683393,410.8528 13.0601393,410.5372 13.0019393,410.1724 L11.9993393,403.7962 L11.6153393,401.3566 C12.5321393,402.9514 14.4893393,405.5518 18.4001393,408.082 C22.3115393,405.5518 24.2675393,402.9514 25.1855393,401.3566 L24.8009393,403.7962 Z M26.5985393,398.0644 C25.7435393,397.87 22.6919393,397.2106 19.9571393,397 L19.9571393,403.4374 L18.4037393,404.5558 L16.8431393,403.4374 L16.8431393,397 C14.1077393,397.2106 11.0561393,397.87 10.2011393,398.0644 C10.0685393,398.0938 9.98213933,398.221 10.0031393,398.3536 L10.8875393,403.969 L11.8913393,410.3446 C12.0071393,411.0796 12.4559393,411.7192 13.1105393,412.0798 L16.8431393,414.1402 L18.4001393,415 L19.9571393,414.1402 L23.6891393,412.0798 C24.3431393,411.7192 24.7925393,411.0796 24.9083393,410.3446 L25.9121393,403.969 L26.7965393,398.3536 C26.8175393,398.221 26.7311393,398.0938 26.5985393,398.0644 L26.5985393,398.0644 Z"
                    id="Defender-Icon" stroke="none" fill="#FFFFFF" fill-rule="evenodd"></path>
        </svg>
		<?php
		$svg = ob_get_clean();

		return 'data:image/svg+xml;base64,' . base64_encode( $svg );
	}

	function remove_settings() {
		delete_site_option( 'wp_defender_shown_activator' );
		delete_site_option( 'wp_defender_is_free_activated' );
		delete_transient( self::CACHE_BLACKLIST_STATUS );
	}

	function remove_data() {
		// TODO: Implement remove_data() method.
	}

	public function data_frontend() {
		list( $endpoints, $nonces ) = Route::export_routes( 'dashboard' );

		return [
			'scan'              => wd_di()->get( Scan::class )->data_frontend(),
			'firewall'          => wd_di()->get( Firewall::class )->data_frontend(),
			'waf'               => wd_di()->get( WAF::class )->data_frontend(),
			'audit'             => wd_di()->get( Audit_Logging::class )->data_frontend(),
			'blacklist'         => [
				'nonces'    => $nonces,
				'endpoints' => $endpoints,
			],
			'two_fa'            => wd_di()->get( Two_Factor::class )->data_frontend(),
			'advanced_tools'    => wd_di()->get( Advanced_Tools::class )->data_frontend(),
			'security_tweaks'   => wd_di()->get( Security_Tweaks::class )->data_frontend(),
			'tutorials'         => wd_di()->get( Tutorial::class )->data_frontend(),
			'show_new_features' => get_site_option( 'wd_show_new_feature' ),
			'notifications'     => wd_di()->get( Notification::class )->data_frontend(),
			'settings'          => wd_di()->get( Main_Setting::class )->data_frontend(),
		];
	}

	public function to_array() {
		// TODO: Implement to_array() method.
	}

	public function import_data( $data ) {
		// TODO: Implement import_data() method.
	}

	/**
	 * @return array
	 */
	public function export_strings() {
		return [];
	}

	/**
	 * Export settings to HUB.
	 * Analog to export_strings but return not array. So separated method.
	 */
	public function export_settings() {
		$config_component = wd_di()->get( \WP_Defender\Component\Backup_Settings::class );
		$data             = $config_component->parse_data_for_hub();
		// Replace all the new line in configs
		$configs = $data['configs'];
		foreach ( $configs as $module => $mdata ) {
			foreach ( $mdata as $key => $value ) {
				if ( is_string( $value ) ) {
					$value         = str_replace( array( "\r", "\n" ), '{nl}', $value );
					$mdata[ $key ] = $value;
				}
			}
			$configs[ $module ] = $mdata;
		}
		$data['configs'] = $configs;
		wp_send_json_success( $data );
	}

	/**
	 * Import settings from HUB.
	 * Analog to import_data but with object $params. So separated method.
	 */
	public function import_settings( $params ) {
		//dirty but quick
		if ( empty( $params->configs ) ) {
			wp_send_json_error( array(
				'message' => __( 'Invalid config', 'wpdef' )
			) );
		}

		$configs = json_decode( json_encode( $params->configs ), true );
		if ( empty( $configs ) ) {
			wp_send_json_error( array(
				'message' => __( 'Empty data', 'wpdef' )
			) );
		}

		$config_component = wd_di()->get( \WP_Defender\Component\Backup_Settings::class );
		$lockout_service  = wd_di()->get( \WP_Defender\Component\Blacklist_Lockout::class );
		foreach ( $configs as $module => $mdata ) {
			foreach ( $mdata as $key => $value ) {
				if ( in_array( $key, array( 'geoIP_db', 'geodb_path') ) ) {
					if ( ! empty( $value ) ) {
						//download it
						$lockout_service->download_geo_ip();
					} else {
						//reset it
						$mdata[ $key ] = '';
					}
				} elseif ( is_string( $value ) ) {
					$value         = str_replace( '{nl}', PHP_EOL, $value );
					$mdata[ $key ] = $value;
				}
			}
			$configs[ $module ] = $mdata;
		}

		//If it's old config structure then we upgrade configs to new format
		if ( ! empty( $configs ) && ! $config_component->check_for_new_structure( $configs ) ) {
			$adapter = wd_di()->get( \WP_Defender\Component\Config\Config_Adapter::class );
			$configs = $adapter->upgrade( $configs );
		}
		$restore_result = $config_component->restore_data( $configs );
		if ( is_string( $restore_result ) ) {
			wp_send_json_error( array(
				'message' => $restore_result
			) );
		}

		wp_send_json_success();
	}
}