<?php

namespace WP_Defender\Behavior\Scan;

use Calotes\Component\Behavior;
use Calotes\Helper\Array_Cache;
use WP_Defender\Behavior\WPMUDEV;
use WP_Defender\Component\Timer;
use WP_Defender\Controller\Scan;
use WP_Defender\Model\Scan_Item;
use WP_Defender\Traits\IO;

class Malware_Scan extends Behavior {
	use IO;

	const YARA_RULES = 'defender_yara_rules', FILE_STUCK = 'defender_scan_stuck';
	/**
	 * Cache for file content
	 * @var string
	 */
	private $content;

	/**
	 * Cache tokens
	 * @var
	 */
	private $tokens;

	/**
	 * Backup memory
	 * @var
	 */
	private $memory;

	public function suspicious_check() {
		set_time_limit( 0 );
		$this->prepare_emergency_shutdown();
		$timer = new Timer();
		$rules = $this->fetch_yara_rules();
		$this->attach_behavior( Malware_Quick_Scan::class, Malware_Quick_Scan::class );
		$this->attach_behavior( Malware_Deep_Scan::class, Malware_Deep_Scan::class );
		$model = $this->owner->scan;
		$pos   = intval( $model->task_checkpoint );

		$files = get_site_option( Gather_Fact::CACHE_CONTENT, [] );
		if ( empty( $files ) ) {
			return true;
		}
		$files = new \ArrayIterator( $files );
		$files->seek( $pos );
		while ( $files->valid() ) {
			if ( ! $timer->check() ) {
				$this->log( 'Rage quit', 'malware_scan' );
				$model->save();
				break;
			}
			if ( $model->is_issue_ignored( $files->current() ) ) {
				$this->log( sprintf( 'skip %s because of file is ignored', $files->current() ), 'malware_scan' );
				$files->next();
				continue;
			}

			list( $result, $qs_detail ) = $this->do_quick_scan( $files->current(), $rules );
			if ( $result ) {
				$this->log( sprintf( 'file %s suspicious', $files->current() ), 'malware_scan' );
				$result = $this->do_deep_scan( $files->current(), $rules, $qs_detail );
				if ( is_array( $result ) ) {
					$result['file'] = $files->current();
					$ret            = $model->add_item( Scan_Item::TYPE_SUSPICIOUS, $result );
				}
			}
			$files->next();
			$model->task_checkpoint = $files->key();
			$model->calculate_percent( $files->key() * 100 / $files->count(), 4 );
			if ( $files->key() % 100 === 0 ) {
				//we should update the model percent each 100 files so we have some progress ont he screen$pos * 100 / $core_files->count()
				$model->save();
			}
		}

		if ( ! $files->valid() ) {
			$last = \WP_Defender\Model\Scan::get_last();
			if ( is_object( $last ) ) {
				$ignored_issues = $last->get_issues( Scan_Item::TYPE_SUSPICIOUS, Scan_Item::STATUS_IGNORE );
				foreach ( $ignored_issues as $issue ) {
					$this->owner->scan->add_item( Scan_Item::TYPE_SUSPICIOUS, $issue->raw_data,
						Scan_Item::STATUS_IGNORE );
				}
			}
			$model->task_checkpoint = null;
		}

		$model->save();

		return ! $files->valid();
	}

	/**
	 * We will use this for a safe switch when memory out happen
	 */
	public function prepare_emergency_shutdown() {
		$this->memory = str_repeat( '*', 1024 * 1024 );

		register_shutdown_function( function () {
			$this->memory = null;
			if ( ( ! is_null( $err = error_get_last() ) ) && ( ! in_array( $err['type'], array(
					E_NOTICE,
					E_WARNING
				) ) ) ) {
				error_log( var_export( $err, true ) );
				$this->log( 'Something wrong happen, saving and quit', 'scan' );
				//$this->owner->scan->status = \WP_Defender\Model\Scan::STATUS_ERROR;
				$this->owner->scan->task_checkpoint += 1;
				$this->owner->scan->save();
			}
		} );
	}

	/**
	 * Fetch yara rules from API
	 *
	 * @return array|mixed
	 */
	private function fetch_yara_rules() {
		$rules = get_site_option( self::YARA_RULES, false );
		if ( is_array( $rules ) ) {
			return $rules;
		}
		$this->attach_behavior( WPMUDEV::class, WPMUDEV::class );
		$rules = $this->make_wpmu_request( WPMUDEV::API_SCAN_SIGNATURE );
		if ( is_array( $rules ) ) {
			$this->log( sprintf( 'Fetched yara rules, total :%s', count( $rules ) ), 'scan_malware' );
			update_site_option( self::YARA_RULES, $rules );

			return $rules;
		}

		return [];
	}
}