<?php

namespace WP_Defender\Behavior\Scan_Item;

use Calotes\Component\Behavior;
use WP_Defender\Component\Error_Code;
use WP_Defender\Model\Scan;
use WP_Defender\Model\Scan_Item;
use WP_Defender\Traits\Formats;
use WP_Defender\Traits\IO;

class Malware_Result extends Behavior {
	use Formats, IO;

	public function to_array() {
		$data            = $this->owner->raw_data;
		$file            = $data['file'];
		$file_created_at = @filemtime( $file );
		if ( $file_created_at ) {
			$file_created_at = $this->format_date_time( $file_created_at );
		} else {
			$file_created_at = 'n/a';
		}
		$file_size = @filesize( $file );
		if ( ! $file_size ) {
			$file_size = 'n/a';
		} else {
			$file_size = $this->format_bytes_into_readable( $file_size );
		}

		return [
			'id'         => $this->owner->id,
			'type'       => Scan_Item::TYPE_SUSPICIOUS,
			'file_name'  => pathinfo( $file, PATHINFO_BASENAME ),
			'full_path'  => $file,
			'date_added' => $file_created_at,
			'size'       => $file_size,
			'short_desc' => __( 'Suspicious function found', 'wpdef' )
		];
	}

	/**
	 * @return array
	 */
	public function ignore() {
		$scan = Scan::get_last();
		$scan->ignore_issue( $this->owner->id );

		return [
			'message' => __( 'The suspicious file has been successfully ignored.', 'wpdef' )
		];
	}

	/**
	 * @return array
	 */
	public function unignore() {
		$scan = Scan::get_last();
		$scan->unignore_issue( $this->owner->id );

		return [
			'message' => __( 'The suspicious file has been successfully restored.', 'wpdef' )
		];
	}

	/**
	 * Delete the file, or whole folder
	 */
	public function delete() {
		$data = $this->owner->raw_data;
		$scan = Scan::get_last();
		$path = $data['file'];
		/**
		 * If the path inside theme or plugin folder, then we remove the whole plugin/theme for
		 * prevent fatal error
		 * If the path is wp-config, wont allow to delete
		 * else where just delete as usual
		 */
		if ( ABSPATH . 'wp-config' === $path ) {
			return new \WP_Error( Error_Code::INVALID,
				__( 'wp-config.php can\'t be removed. Please remove the suspicious code manually.', 'wpdef' ) );
		}

		$plugin_path = WP_CONTENT_DIR . '/plugins';
		$theme_path  = WP_CONTENT_DIR . '/themes';
		$rev_path    = str_replace( [ $plugin_path, $theme_path ], '', $path );
		//if it not relative path, which mean it is something lay on wp-content folder, we can delete it
		if ( file_exists( $rev_path ) ) {
			@unlink( $rev_path );
		}
		$pools = explode( '/', $rev_path );
		$pools = array_filter( $pools );
		if ( 1 === count( $pools ) ) {
			//this mean the file is hang in plugins/ or themes/ in first level, delete it
			@unlink( $path );
		} else {
			//this is inside a folder, we delete whole
			$folder_path = substr( $path, 0, strrpos( $path, $pools[0] ) );
		}
		$scan->remove_issue( $this->owner->id );

		return [
			'message' => __( 'This item has been permanently removed', 'wpdef' )
		];
	}

	/**
	 *
	 */
	public function pull_src() {
		$data = $this->owner->raw_data;
		if ( ! file_exists( $data['file'] ) ) {
			return [
				'origin' => '',
				'mapper' => array()
			];
		}
		$markers = [];
		foreach ( $data as $item ) {
			if ( isset( $item['catches'] ) ) {
				foreach ( $item['catches'] as $catch ) {
					$markers = array_merge( $markers, $catch['mapper'] );
				}
			}
		}
		wp_send_json_success( [
			'origin' => file_get_contents( $data['file'] ),
			'mapper' => $markers,
		] );
	}
}